/* Copyright (C) 2012-2018 RealVNC Ltd.  All Rights Reserved. */

#ifndef __VNCAUDIOROUTER_H__
#define __VNCAUDIOROUTER_H__

/* You will need to define exactly one of VNC_USE_STDINT_H, or
 * VNC_USE_DEFAULT_INT_TYPEDEFS, otherwise this will not compile */
#include <vncviewersdk.h>


/**
* \file vncaudiorouter.h
*
* This is the main header file for the VNC Audio Router API.
*
* This file defines the VNC Audio Router API Interface, the data
* structure associated with it and the sole entry point for the Audio
* Router DLL or shared object.
*
* \see VNCAudioRouterInitialize, VNCAudioRouterSDK
*/

/**
 * \mainpage VNC Audio Router API
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_memory
 *  -# \ref section_thread
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * The VNC Audio Router is used from the Viewer side of an RFB4 or MirrorLink&reg;
 * connection to a mobile device. The Router is built on top of the VNC
 * Viewer and VNC Discovery SDKs and abstracts away from the APIs used
 * in preparing audio connections. Therefore the usage of the Audio Router
 * API is the same, regardless of whether the Viewer is connected to an RFB4
 * device, or a MirrorLink device. 
 *
 * To find out more about the VNC Viewer SDK, or the VNC Discovery SDK please
 * read the documentation describing their client API. The Viewer SDK
 * documentation is entitled "VNC Mobile Viewer SDK - Native Interface". 
 * For the Discovery SDK please read "VNC Discovery SDK - Native Interface".
 *
 * An instance of the Audio Router needs to be created for each viewer
 * instance.  The Router can run in one of the two modes at any one time: RFB4 
 * or MirrorLink, and in each mode it can communicate with one device at any
 * one time. Both the mode and the device can be changed on the router.
 * 
 * When connected to an RFB4 device, the Router uses its extension message 
 * APIs. Therefore using the VNC Audio Router depends on the correct 
 * use of the VNC Viewer SDK to create a VNC session between the Viewer and a 
 * VNC Server running on a mobile device.
 *
 * When connected to a MirrorLink device, the Router will use the Discovery SDK
 * functionality to set up audio connections to the MirrorLink server. This is
 * done by using a MirrorLink Discoverer plug-in that follows the specific 
 * behaviour described within the VNC Discovery APIs.
 *
 * The Audio Router allows a Viewer application to obtain information
 * on available protocols (Bluetooth/RTP) on the device. Information about
 * these protocols can be retrieved through the Router. This includes existing 
 * Bluetooth connections between the mobile device and a Bluetooth
 * audio device, RTP formats supported, etc. It also allows the Viewer
 * application to request the set-up, or termination, of an audio connection
 * between the mobile device and the desired audio device.
 *
 * Although the Router presents this functionality to the Viewer application,
 * it may be possible that the device side (either VNC Server, or MirrorLink
 * server) does not support it.
 *
 * Some mobile platforms might not have the necessary functionality that
 * enables the completion of these requests automatically. The Viewer
 * application will be notified of any steps that are necessary for the user
 * to take for ensuring the successful creation of an audio connection.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_initializing Initializing the SDK
 *
 * To be able to use the Audio Router functionality, a Viewer
 * application needs first to initialize the SDK through
 * VNCAudioRouterInitialize(). This will provide the SDK functions that can be
 * called from the application. Also, it needs to create an Audio Router for the
 * Viewer that it uses, through VNCAudioRouterCreate(). The Router should be created
 * after the Viewer has been created, but before a Viewer session is requested (before
 * the Viewer thread is created). This is so that the Audio Router can register its
 * extension with the Viewer. The Audio Router uses the extension message functionality
 * of the Viewer SDK to establish the audio links.
 *
 * The lifetime of the Audio Router should be the same as the lifetime of the
 * Viewer that it uses: the Router should be destroyed just before the Viewer.
 * This is because only one extension of a given name can be registered with each
 * Viewer. Therefore, destroying the Router early and recreating it for the same
 * Viewer, will result in failure, since the new Router will not be able to 
 * re-register the extension.
 *
 * When creating the Audio Router, the application can register the callbacks
 * it wishes to receive. If it does not want to be notified for some events, it
 * can leave those callback entries as NULL.
 *
 * \subsection subsection_usage_configuring Configuring the Audio Router
 *
 * When intending to communicate to a device, the router should be set in one
 * of two modes: RFB4 (VNCAudioRouterUseRfb4Extension()), or MirrorLink
 * (VNCAudioRouterUseMirrorLinkDevice()).
 * When connecting to a device with an RFB4 server, the first method should be 
 * used. When connecting to a MirrorLink device that has
 * been discovered through the Discovery SDK, the second method should be used.
 * In that case the Discovery SDK, the Discoverer and Entity instances need to
 * be passed in, as they will be used to communicate with the device through the
 * MirrorLink UPnP mechanisms. The Audio Router uses the Discovery SDK to
 * establish the audio links with the device; so in the MirrorLink case, the
 * Entity must have originated from the Discovery SDK.
 *
 * \subsection subsection_usage_retrieving Retrieving information about the device
 *
 * Once the usage is set, the Audio Router should be queried about the available
 * protocols on the device (through VNCAudioRouterGetAvailableProtocols()). This
 * will take the necessary steps to communicate with the device and it will
 * return the available protocols to the client through a callback
 * (VNCAudioRouterProtocolsAvailableCb()).
 *
 * Once the available protocols are received. The application can then 
 * retrieve information about the protocol parameters on the device, by calling
 * VNCAudioRouterGetRemotingInfo(). Then the client can decide which of the available
 * protocols it wishes to use. It will also need to configure some parameters through
 * VNCAudioRouterSetParameter().
 *
 * For Bluetooth they are VNCAudioRouterParameterKeyBtStart (optional - default
 * server), VNCAudioRouterParameterKeyBtAddress (mandatory, for server starting
 * the Bluetooth connection) and VNCAudioRouterParameterKeyBtPin (optional).
 * The Bluetooth start option specifies which application - the server or the 
 * client - will initiate the Bluetooth connection. Not all servers support 
 * both, so you can find out from the remoting info what the server supports. 
 * If the server is configured to start the connection, then a Bluetooth 
 * address is necessary. That will be given to the server and 
 * connection-initiation will be entirely handled by the server. If the client 
 * should initiate the connection, then the server will be configured to start 
 * Bluetooth and accept incoming connections. A Bluetooth address will be 
 * retrieved from the server side. The configured Bluetooth address must be of
 * the form 00:11:22:AA:BB:CC (all letters capital) and should be null
 * terminated. The PIN should be null terminated and not exceed 64 characters.
 *
 * For RTP, the parameters are VNCAudioRouterParameterKeyRtpPayloadType,
 * VNCAudioRouterParameterKeyRtpAudioIpl and 
 * VNCAudioRouterParameterKeyRtpAudioMpl. They are all optional and have the
 * same meaning as in the MirrorLink specification.
 *
 * \subsection subsection_usage_connecting Starting an audio connection
 *
 * Once the configuration is made, the application can request the creation of
 * the audio links. This is done through VNCAudioRouterAddLinks(). Success is
 * reported through VNCAudioRouterLinkAddedCb(), and failure through
 * VNCAudioRouterLinkAddedCb(). In the case of success, some extra information
 * will be provided through the callback. For example, for RTP, the extra
 * information is the address to which the Viewer needs to connect for
 * receiving/sending UDP RTP packets (please note that, as per the MirrorLink
 * specification, a UDP packet needs to be sent to that address to start the
 * streaming). For Bluetooth, the extra information will be the Bluetooth
 * address of the device. 
 *
 * The application also has the possiblity of requesting information about the
 * existing Bluetooth connection. This is so that it can inform the user to
 * take appropiate action for setting up Bluetooth connections if it does not
 * want automatically to do so. This is done through
 * VNCAudioRouterCheckBtConnection() and the information is provided in the
 * VNCAudioRouterCheckBtConnectionCb() callback.
 *
 * \subsection subsection_usage_disconnecting Disconnecting
 *
 * The application may also request the termination of the audio connection for
 * one or more of the protocols available. This is done through the
 * VNCAudioRouterRemoveLinks() call, with the outcome reported in
 * VNCAudioRouterLinkRemovedCb(), or VNCAudioRouterLinkRemovalFailedCb().
 *
 * If at any point the device becomes unavailable (the RFB4 VNC session has closed,
 * or the MirrorLink server is disconnected), then
 * VNCAudioRouterDeviceNotAvailableCb() will inform the application of this. The
 * application should re-request VNCAudioRouterGetAvailableProtocols(), which will
 * provide available protocols when a connection to the server is re-established
 * (a new RFB4 connection). If the Router is not able to wait for
 * available protocols, then an error will be reported.
 *
 * The client application can, at any point, change the usage of the router
 * (from RFB4 to MirrorLink, or vice-versa). This can happen if the application
 * becomes interested in another device. Please note that for MirrorLink, the
 * device should be reset every time through
 * VNCAudioRouterUseMirrorLinkDevice(), since this takes an entity, which
 * represents a server. For RFB4, since the Viewer can only have one session
 * and the extension message functionality will be used, the Router will 
 * always use the device the Viewer has a connection to.
 *
 * \subsection subsection_usage_uninitializing Uninitializing the SDK
 *
 * When the Audio Router is no longer needed, VNCAudioRouterDestroy()
 * should be called to free any resources. Destroy can be called while there
 * are outstanding requests. Those requests will never complete once destroy
 * has been called.
 *
 * \section section_memory Memory management
 *
 * The Audio Router is owned by the client application, which has the
 * responsibility of calling VNCAudioRouterDestroy() when no longer needed. This
 * call will free any memory held by the Router.
 *
 * During the API calls and during the callback calls ownership is not
 * transferred. Therefore the client application needs to free its own
 * resources, even if it passed them to the Router APIs. Also the client
 * application should create copies of any data structures that are passed in
 * the callbacks if it wishes to use them outside the callback.
 *
 * The Audio Router holds the pointers to the Viewer SDK and Viewer
 * instance as they were passed in VNCAudioRouterCreate(), so the
 * underlying data structures should continue to be present at the same
 * location throughout the lifetime of the Router. Also, in the MirrorLink
 * usage, the Router holds the pointers to the Discovery SDK and Discoverer as
 * they were given in VNCAudioRouterUseMirrorLinkDevice().
 *
 * \section section_thread Threading model
 *
 * The Router will not create its own threads. It will handle calls in the 
 * same thread as the caller thread. Therefore all the API requests will run 
 * in the thread from which the client application made the request. If the
 * calls are successful, then the callbacks will be created from within
 * callbacks of the Viewer SDK, or Discovery SDK (depending on the usage set).
 * So, that will happen in the respective SDK thread. Therefore all the
 * callbacks from the router will be on the thread of the SDK it uses.
 *
 * If the Router is not able to send a request to the device, then it will 
 * return an error straight away and no callbacks will be received.
 *
 * The log callbacks can be received from any thread, but the audio router will
 * ensure that there is no race condition between them. This is done through a
 * mutex/critical section. For this reason, to avoid deadlocks, the client 
 * application should not make any calls to the Audio Router or any of the SDKs
 * from within the logging callback.
 *
 * \section section_legal Legal Information
 *
 * Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 * MirrorLink is a registered trademark of Car Connectivity Consortium LLC.
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterInitialize, VNCAudioRouterCallbacks
 */


#ifdef __cplusplus
extern "C"
{
#endif

/** 
 * The type for the protocols. This can have the value of
 * 'undefined', a bitmask containing one or more protocols, or have the value of
 * 'any'.
 *
 * \see VNCAudioRouterProtocolUndefined, VNCAudioRouterProtocolBtHfp,
 * VNCAudioRouterProtocolBtA2dp, VNCAudioRouterProtocolRtpOut,
 * VNCAudioRouterProtocolRtpIn, VNCAudioRouterProtocolAny,
 * VNCAudioRouterProtocolRtpVoiceCommand
 */
typedef vnc_uint32_t VNCAudioRouterProtocols;

/**
 * The undefined profile. This is used in the callbacks of the API to mean that
 * no protocol, connection, or bonding, is available/exists between the target
 * device and an audio device.
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolUndefined   0

/**
 * The Bluetooth Handsfree profile.
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolBtHfp       2

/**
 * The Bluetooth A2DP profile.
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolBtA2dp      4

/**
 * The RTP Out protocol (same meaning as in the MirrorLink specification - RTP
 * from the MirrorLink server to the client).
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolRtpOut      8

/**
 * The RTP In protocol (same meaning as in the MirrorLink specification - RTP
 * from the MirrorLink client to the server).
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolRtpIn       16

/**
 * The RTP Voice Command protocol which uses the same mechanism as the RTP In
 * protocol, but communicates with a Voice Command Engine application (see the
 * MirrorLink specification for information on the Voice Command Engine). It is
 * represented by a separate protocol for simplicity.
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolRtpVoiceCommand 32

/**
 * The 'any' protocol. This is used by itself, rather than combined with other
 * protocols in a bitmask. In the callback APIs it means
 * 'connection/bonding is present, but can't determine the exact profile'.
 *
 * \see VNCAudioRouterProtocols
 */
#define VNCAudioRouterProtocolAny 0xFFFFFFFF

/** The type for the errors reported by the Audio Router API. */
typedef vnc_uint16_t VNCAudioRouterError;

/** No error reported. */
#define VNCAudioRouterErrorNone                 0
/** Error reported by the mobile device. Bluetooth is not supported on the
 * device. */
#define VNCAudioRouterErrorBtNotSupported       1
/** Error reported by the mobile device. The mobile platform can not complete the
 * request automatically. Only a manual check/connect/disconnect can be done. */
#define VNCAudioRouterErrorManualOnly           2
/** Error reported by the mobile device. Pairing failed because of an
 * authentication failure. */
#define VNCAudioRouterErrorAuthFailure          3
/** Error reported by the mobile device. User is needed to manually complete
 * the request. */
#define VNCAudioRouterErrorUserNeeded           4
/** Error reported by the mobile device. The protocol is not supported. */
#define VNCAudioRouterErrorProtocolNotSupported 5
/** Error reported by the mobile device. The address provided is not valid. */
#define VNCAudioRouterErrorInvalidAddress       6
/** Error reported by the mobile device. The requested device can not be found. */
#define VNCAudioRouterErrorDeviceNotFound       7
/** Error reported by the mobile device. The PIN needs to be entered by the
 * user on the mobile device. */
#define VNCAudioRouterErrorManualPinNeeded      8
/** Error reported by the mobile device. Can not pair and/or connect
 * automatically. The user must create the pairing and/or connect manually. */
#define VNCAudioRouterErrorManualPairingNeeded  9
/** Error reported by the mobile device. Can not determine the
 * connected/bonded profile. */
#define VNCAudioRouterErrorProfileNotDetermined 10
/** Error reported by the mobile device. Can not connect all the requested
 * profiles, only some of them. */
#define VNCAudioRouterErrorPartialConnect       11
/** Error reported by the mobile device. Bluetooth is switched off, so the
 * request can not be completed. */
#define VNCAudioRouterErrorBtOff                12
/** Error reported by the mobile device. The audio device is paired with the 
 * target device, but is not set as authorized (fully bonded). */
#define VNCAudioRouterErrorNotAuthorized        13
/** Unable to find an item which is needed to complete the request. */
#define VNCAudioRouterErrorNotFound             100
/** At least one of the parameters passed to the API is invalid. */
#define VNCAudioRouterErrorInvalidParameter     101
/** Not enough memory to complete the request. */
#define VNCAudioRouterErrorOutOfMemory          102
/** Request is not supported. */
#define VNCAudioRouterErrorNotSupported         103
/** An item/connection/request already exists */
#define VNCAudioRouterErrorAlreadyExists        104
/** Request can't be completed because the configuration step wasn't completed
 * successfully. */
#define VNCAudioRouterErrorNotConfigured        105
/** Can't make the request because of a disconnection. */
#define VNCAudioRouterErrorDisconnected         106
/** Not enough buffer space to send message out to the VNC server. */
#define VNCAudioRouterErrorInsufficientBufferSpace 107
/** The VNC connection is not started, so it is not possible to send the
 * request at the moment. */
#define VNCAudioRouterErrorNotStarted           108
/** One of the string arguments is bigger than the maximum expected size. */
#define VNCAudioRouterErrorTooBig               109
/** There is a conflict between the settings requested. */
#define VNCAudioRouterErrorConflict             110
/** The request timed out. */
#define VNCAudioRouterErrorTimedOut             111
/** Unable to make request because the router is shutting down. */
#define VNCAudioRouterErrorShuttingDown         112
/** Unable to make request because no usage has been set. */
#define VNCAudioRouterErrorUsageNotSet          113
/** The audio router extension feature is not licensed. */
#define VNCAudioRouterErrorNotLicensed          114
/** A general, unspecified, error occurred. */
#define VNCAudioRouterErrorGeneral              0xFFFF

/**
 * \brief The structure containing the data needed by the Audio Router.
 *
 * This is the structure that contains the data for the Audio Router.
 * It is created and populated by either VNCAudioRouterCreate.
 *
 * The structure is opaque to the client application and it should only be used
 * for passing into the Audio Router API calls.
 *
 * The client application owns the data structure and it should clear any
 * resources taken by it by calling VNCAudioRouterDestroy.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterInitialize,
 * VNCAudioRouterCallbacks, VNCAudioRouterCreate, 
 * VNCAudioRouterDestroy
 */
typedef struct VNCAudioRouter VNCAudioRouter;

/** 
 * The key used for finding out whether the server can initiate the Bluetooth
 * connection, the client has to, or both are supported.
 * \see VNCAudioRouterRemotingInfoBtStartClient, 
 *    VNCAudioRouterRemotingInfoBtStartServer, 
 *    VNCAudioRouterRemotingInfoBtStartBoth, VNCAudioRouterGetRemotingInfo
 */
#define VNCAudioRouterRemotingInfoKeyBtStart "btStart"

/**
 * The client must initiate the Bluetooth connection.
 *
 * \see VNCAudioRouterRemotingInfoKeyBtStart
 */
#define VNCAudioRouterRemotingInfoBtStartClient "1"

/**
 * The server must initiate the Bluetooth connection.
 *
 * \see VNCAudioRouterRemotingInfoKeyBtStart
 */
#define VNCAudioRouterRemotingInfoBtStartServer "2"

/**
 * Both client and server can initiate the Bluetooth connection.
 *
 * \see VNCAudioRouterRemotingInfoKeyBtStart
 */
#define VNCAudioRouterRemotingInfoBtStartBoth "1,2"

/**
 * The key for retrieving the RTP Out formats supported by the server.
 *
 * \see VNCAudioRouterGetRemotingInfo
 */
#define VNCAudioRouterRemotingInfoKeyRtpOutFormat "rtpOutFormat"

/**
 * The key for retrieving the RTP In formats supported by the server.
 *
 * \see VNCAudioRouterGetRemotingInfo
 */
#define VNCAudioRouterRemotingInfoKeyRtpInFormat "rtpInFormat"

/**
 * The key for retrieving the RTP Voice Command formats supported by the server.
 *
 * \see VNCAudioRouterGetRemotingInfo
 */
#define VNCAudioRouterRemotingInfoKeyRtpVoiceCommandFormat "rtpVoiceCommandFormat"

/**
 * The key for setting the Bluetooth start preference. Either the client, or
 * server must initiate the connection.
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter,
 * VNCAudioRouterParameterBtStartClient, VNCAudioRouterParameterBtStartServer
 */
#define VNCAudioRouterParameterKeyBtStart "btStart"

/**
 * The value indicating that the client will start the Bluetooth connection.
 *
 * \see VNCAudioRouterParameterKeyBtStart
 */
#define VNCAudioRouterParameterBtStartClient "1"

/**
 * The value indicating that the server will start the Bluetooth connection.
 *
 * \see VNCAudioRouterParameterKeyBtStart
 */
#define VNCAudioRouterParameterBtStartServer "2"

/**
 * The key for setting the Bluetooth address. It is mandatory when the server
 * starts the Bluetooth connection. It must be NULL terminated and of the form
 * 00:11:22:AA:BB:CC (17 characters long plus the NULL termination, all letters
 * in capital).
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter
 */
#define VNCAudioRouterParameterKeyBtAddress "btAddress"

/**
 * The key for setting the Bluetooth PIN. It is useful when the server starts
 * the Bluetooth connection. Must be NULL terminated and not exceed 64
 * characters (together with the NULL termination).
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter
 */
#define VNCAudioRouterParameterKeyBtPin "btPin"

/**
 * The key for setting the RTP payload type (sometimes referred to as RTP
 * format). It is a comma separated list of supported formats (like
 * "0,98,99"). See the MirrorLink specification for more information.
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter
 */
#define VNCAudioRouterParameterKeyRtpPayloadType "rtpPayloadType"

/**
 * The key for setting the RTP Audio IPL. Must be a string representation of
 * a number. See the MirrorLink specification for more information.
 * The server must initiate the connection.
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter
 */
#define VNCAudioRouterParameterKeyRtpAudioIpl "rtpAudioIpl"

/**
 * The key for setting the RTP Audio MPL. Must be a string representation of
 * a number. See the MirrorLink specification for more information.
 * The server must initiate the connection.
 *
 * \see VNCAudioRouterSetParameter, VNCAudioRouterGetParameter
 */
#define VNCAudioRouterParameterKeyRtpAudioMpl "rtpAudioMpl"

/**
 * Critical log severity.
 *
 * \see VNCAudioRouterLog, VNCAudioRouterSetLoggingSeverity
 */
#define VNCAudioRouterLogSeverityCritical 0

/**
 * Error log severity.
 *
 * \see VNCAudioRouterLog, VNCAudioRouterSetLoggingSeverity
 */
#define VNCAudioRouterLogSeverityError 10

/**
 * Warning log severity.
 *
 * \see VNCAudioRouterLog, VNCAudioRouterSetLoggingSeverity
 */
#define VNCAudioRouterLogSeverityWarning 20

/**
 * Info log severity.
 *
 * \see VNCAudioRouterLog, VNCAudioRouterSetLoggingSeverity
 */
#define VNCAudioRouterLogSeverityInfo 30

/**
 * Debug log severity.
 *
 * \see VNCAudioRouterLog, VNCAudioRouterSetLoggingSeverity
 */
#define VNCAudioRouterLogSeverityDebug 100

/**
 * \brief Structure containing extra info for Bluetooth "start client"
 * connections.
 *
 * This structure represents the information passed in the link/remove link
 * callbacks for Bluetooth protocols when the client is expected to start the
 * connection.
 *
 * \see VNCAudioRouterLinkAddedCb, VNCAudioRouterLinkFailedCb,
 * VNCAudioRouterLinkRemovedCb, VNCAudioRouterLinkRemovalFailedCb
 */
typedef struct
{
  /** The URI representing the Bluetooth address of the MirrorLink server. */
  const char *serverBtAddress;
} VNCAudioRouterExtraInfoBtStartClient;

/**
 * \brief Structure containing extra info for Bluetooth "start server"
 * connections.
 *
 * This structure represents the information passed in the link/remove link
 * callbacks for Bluetooth protocols when the server is expected to start the
 * connection.
 *
 * \see VNCAudioRouterLinkAddedCb, VNCAudioRouterLinkFailedCb,
 * VNCAudioRouterLinkRemovedCb, VNCAudioRouterLinkRemovalFailedCb
 */
typedef struct
{
  /** The URI representing the Bluetooth address the MirrorLink server
   * connected to. */
  const char *clientBtAddress;
} VNCAudioRouterExtraInfoBtStartServer;

/**
 * \brief Structure containing extra info for RTP Out connections.
 *
 * This structure represents the information passed in the link/remove link
 * callbacks for RTP Out protocol.
 *
 * \see VNCAudioRouterLinkAddedCb, VNCAudioRouterLinkFailedCb,
 * VNCAudioRouterLinkRemovedCb, VNCAudioRouterLinkRemovalFailedCb
 */
typedef struct
{
  /** The URI of the RTP server. The MirrorLink client is expected to send a
   * an UDP packet containing 1 byte (of arbitrary value) to the URI (a 'hello'
   * message), before the RTP server can start streaming (see the MirrorLink
   * specification for more information. */
  const char *uri;
} VNCAudioRouterExtraInfoRtpOut;

/**
 * \brief Structure containing extra info for RTP In and RTP Voice Command 
 * connections.
 *
 * This structure represents the information passed in the link/remove link
 * callbacks for RTP In and RTP Voice Command protocols (RTP Voice Command
 * works in the same way as RTP In).
 *
 * \see VNCAudioRouterLinkAddedCb, VNCAudioRouterLinkFailedCb,
 * VNCAudioRouterLinkRemovedCb, VNCAudioRouterLinkRemovalFailedCb
 */
typedef struct
{
  /** The URI where to send the RTP packets */
  const char *uri;
  /** The payload types supported by the RTP client (MirrorLink server
   * application). The viewer application should use one of the payload types
   * defined here */
  const char *payloadTypes;
  /** The audio Initial Playback Latency requested by the RTP client
   * (MirrorLink server). This is the audioIpl as received from the server. If
   * empty, or NULL, the default value, as specified by the MirrorLink
   * specification, is assumed.*/
  const char *audioIpl;
  /** The audio Maximum Payload Length requested by the RTP client
   * (MirrorLink server). This is the audioMpl as received from the server. If
   * empty, or NULL, the default value, as specified by the MirrorLink
   * specification, is assumed.*/
  const char *audioMpl;
} VNCAudioRouterExtraInfoRtpIn;

/**
 * \brief Callback notifying the application that the device is no longer
 * available.
 *
 * Callback notifying the client application that the device with which the
 * Router has been in communication is no longer available. In response, the
 * client application should call VNCAudioRouterGetAvailableProtocols, in order
 * to be notified of protocols becoming available again.
 *
 * \param pContext The client application context.
 *
 * \see VNCAudioRouterCallbacks
 */
typedef void VNCCALL VNCAudioRouterDeviceNotAvailableCb(void *pContext);

/**
 * \brief Callback returning the available protocols.
 *
 * This is a response to VNCAudioRouterGetAvailableProtocols. Successfully
 * retrieved protocols are provided to the client application by this callback.
 *
 * \param pContext The client application context.
 * \param protocols The bitmask representing the available protocols. If no
 * protocols are available, this will be set to VNCAudioRouterProtocolUndefined.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterGetAvailableProtocols
 */
typedef void VNCCALL VNCAudioRouterProtocolsAvailableCb(void *pContext,
                                        VNCAudioRouterProtocols protocols);

/**
 * \brief Failure callback for the get available protocols request.
 *
 * This is a response to VNCAudioRouterGetAvailableProtocols. If the SDK
 * reports some error and the protocols cannot be retrieved, this callback
 * will pass the converted error to the client.
 *
 * \param pContext The client application context.
 * \param error The error encountered when retrieving protocols.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterGetAvailableProtocols
 */
typedef void VNCCALL VNCAudioRouterProtocolCheckErrorCb(void *pContext,
                                        VNCAudioRouterError error);

/**
 * \brief Callback reporting the Bluetooth connection status.
 *
 * This is the callback for VNCAudioRouterCheckBtConnection. It reports
 * information on the current Bluetooth state (on/off), if the devices are
 * bonded and if they are connected. If bonding and connection exists between
 * the devices, then the bonded and connected profiles are returned.
 *
 * On some mobile devices it is not possible to determine some of the
 * information that is intended to be returned here. For those situations,
 * there are error codes that clarify how accurate the information is.
 *
 * \param pContext The context given by the client application.
 * \param error The outcome of attempting to send the check request to the
 *      mobile device. If this is VNCAudioRouterErrorNone then the request
 *      was sent successfully. Otherwise, all the other errors in the callback
 *      will be set to VNCAudioRouterErrorGeneral.
 * \param btError The error related to checking the current Bluetooth state of
 *      the mobile device. 
 *        - VNCAudioRouterErrorNone if checking whether Bluetooth is on or 
 *        off was successful.
 *        - VNCAudioRouterErrorBtNotSupported if Bluetooth is not supported by
 *        the device. 
 *        - VNCAudioRouterErrorManualOnly if it is not possible to check the 
 *        Bluetooth state on the device.
 *        - VNCAudioRouterErrorGeneral means any other unspecified error occurred.
 *        - Other errors could be returned in future implementations
 * \param btEnabled Specifies whether Bluetooth is switched on or off on the
 *      device, if btError is VNCAudioRouterErrorNone. If it is set to 0 then
 *      Bluetooth is off, otherwise Bluetooth on the mobile device is on.
 * \param bondedProfilesError The error related to checking whether the mobile
 *      device is bonded (paired and trusted) with the required Bluetooth
 *      device. 
 *        - VNCAudioRouterErrorNone if bonding was checked successfully. 
 *        - VNCAudioRouterErrorBtNotSupported if Bluetooth is not supported 
 *        by the mobile device.
 *        - VNCAudioRouterErrorManualOnly if it is not possible to 
 *        automatically check if the devices are bonded. 
 *        - VNCAudioRouterErrorProfileNotDetermined if the devices are 
 *        bonded, but the mobile can not tell which profiles they are bonded 
 *        on. 
 *        - VNCAudioRouterErrorBtOff if Bluetooth is off and on the mobile 
 *        device it is not possible to check bonding, unless Bluetooth is 
 *        switched on. 
 *        - VNCAudioRouterErrorNotAuthorized if the devices are paired, but not
 *        authorized (paired, but not bonded).
 *        - VNCAudioRouterErrorGeneral means any other unspecified error 
 *        occurred.
 *        - Other errors could be returned in future implementations.
 * \param bondedBtProfiles Specifies with which profiles the mobile and the
 *      Bluetooth device are bonded. If the bondedProfilesError signals any
 *      error other than VNCAudioRouterErrorProfileNotDetermined, and
 *      VNCAudioRouterErrorNotAuthorized, this will be
 *      VNCAudioRouterProtocolUndefined. If the bondedProfilesError is
 *      VNCAudioRouterErrorProfileNotDetermined then this will be
 *      VNCAudioRouterProtocolAny. If there is no error, then this will be
 *      a bitmask of the bonded profiles. If the devices are not bonded, then
 *      this will be VNCAudioRouterProtocolUndefined.
 * \param connectedProfilesError The error related to checking whether the mobile
 *      device is connected with the required Bluetooth audio 
 *      device. 
 *        - VNCAudioRouterErrorNone if the connection was checked 
 *        successfully. 
 *        - VNCAudioRouterErrorBtNotSupported if Bluetooth is not supported 
 *        by the mobile device. 
 *        - VNCAudioRouterErrorManualOnly if it is not possible to 
 *        automatically check if the devices are connected. 
 *        - VNCAudioRouterErrorProfileNotDetermined if the devices 
 *        are connected, but the mobile can not tell which profiles they are 
 *        connected on. 
 *        - VNCAudioRouterErrorGeneral means any other unspecified error 
 *        occurred. 
 *        - If Bluetooth is off, then VNCAudioRouterErrorNone is returned 
 *        because it is implicitly true that there is not Bluetooth connection.
 *        - Other errors could be returned in future implementations, for example
 *        once MirrorLink will be supported.
 * \param connectedBtProfiles Specifies with which profiles the mobile and the
 *      Bluetooth device are connected. If the connectedProfilesError signals an error (except
 *      VNCAudioRouterErrorProfileNotDetermined) this will be
 *      VNCAudioRouterProtocolUndefined. If the connectedProfilesError is
 *      VNCAudioRouterErrorProfileNotDetermined then this will be
 *      VNCAudioRouterProtocolAny. If Bluetooth is off, then this will be 
 *      VNCAudioRouterProtocolUndefined. If there is no error, then this will be
 *      a bitmask of the connected profiles. If the devices are not connected this
 *      will be VNCAudioRouterProtocolUndefined.
 * \param btAddress Confirms the Bluetooth address of the audio device for
 *      which the check was done. Owned by the Audio Router.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterCheckBtConnection
 */
typedef void VNCCALL VNCAudioRouterCheckBtConnectionCb(
                    void *pContext,
                    VNCAudioRouterError error,
                    VNCAudioRouterError btError,
                    vnc_int32_t btEnabled,
                    VNCAudioRouterError bondedProfilesError,
                    VNCAudioRouterProtocols bondedBtProfiles,
                    VNCAudioRouterError connectedProfilesError,
                    VNCAudioRouterProtocols connectedBtProfiles,
                    const char *btAddress);

/**
 * \brief Successful response to the add links request.
 *
 * This is the callback used when a call to VNCAudioRouterAddLinks is
 * successful. It contains the protocols that are connected/ready for
 * connection. 
 *
 * \param pContext The context given by the client application.
 * \param protocols The connected protocols. Can be a bitmask.
 * \param extraInfo Pointer to a data structure containing extra information
 *    about the linked protocol. The underlying structure differs for each
 *    protocol/type of connection. 
 *     - For a Bluetooth protocol and "start client" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartClient.
 *     - For a Bluetooth protocol and "start server" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartServer.
 *     - For an RTP Out protocol the structure is of type
 *     VNCAudioRouterExtraInfoRtpOut.
 *     - For RTP In and Voice Command protocols, the structure is of type
 *     VNCAudioRouterExtraInfoRtpIn.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterAddLinks,
 * VNCAudioRouterExtraInfoBtStartClient, VNCAudioRouterExtraInfoBtStartServer,
 * VNCAudioRouterExtraInfoRtpOut, VNCAudioRouterExtraInfoRtpIn
 */
typedef void VNCCALL VNCAudioRouterLinkAddedCb(
                    void *pContext,
                    VNCAudioRouterProtocols protocols,
                    const void *extraInfo);

/**
 * \brief Error response to the add links request.
 *
 * This is the callback used when a call to VNCAudioRouterAddLinks is
 * not successful for some protocols. It contains the protocols that 
 * failed to connect. 
 *
 * \param pContext The context given by the client application.
 * \param protocols The protocols that failed. Can be a bitmask.
 * \param error The error reported.
 * \param extraInfo Pointer to a data structure containing extra information
 *    about the failed linked protocol. The underlying structure differs for each
 *    protocol/type of connection. 
 *     - For a Bluetooth protocol and "start client" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartClient.
 *     - For a Bluetooth protocol and "start server" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartServer.
 *     - For an RTP Out protocol the structure is of type
 *     VNCAudioRouterExtraInfoRtpOut.
 *     - For RTP In and Voice Command protocols, the structure is of type
 *     VNCAudioRouterExtraInfoRtpIn.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterAddLinks
 * VNCAudioRouterExtraInfoBtStartClient, VNCAudioRouterExtraInfoBtStartServer,
 * VNCAudioRouterExtraInfoRtpOut, VNCAudioRouterExtraInfoRtpIn
 */
typedef void VNCCALL VNCAudioRouterLinkFailedCb(
                    void *pContext,
                    VNCAudioRouterProtocols protocols,
                    VNCAudioRouterError error,
                    const void *extraInfo);

/**
 * \brief Notifies that the link for some protocols has been removed.
 *
 * This is a successful response to the remove links request, or sent as
 * a result of a server decision to close the audio links.
 *
 * This may be a callback used when a call to VNCAudioRouterRemoveLinks is
 * successful. It contains the protocols that were disconnected.
 *
 * \param pContext The context given by the client application.
 * \param protocols The disconnected protocols. Can be a bitmask.
 * \param extraInfo Pointer to a data structure containing extra information
 *    about the unlinked protocol. The underlying structure differs for each
 *    protocol/type of connection. 
 *     - For a Bluetooth protocol and "start client" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartClient.
 *     - For a Bluetooth protocol and "start server" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartServer.
 *     - For an RTP Out protocol the structure is of type
 *     VNCAudioRouterExtraInfoRtpOut.
 *     - For RTP In and Voice Command protocols, the structure is of type
 *     VNCAudioRouterExtraInfoRtpIn.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterRemoveLinks
 * VNCAudioRouterExtraInfoBtStartClient, VNCAudioRouterExtraInfoBtStartServer,
 * VNCAudioRouterExtraInfoRtpOut, VNCAudioRouterExtraInfoRtpIn
 */
typedef void VNCCALL VNCAudioRouterLinkRemovedCb(
                    void *pContext,
                    VNCAudioRouterProtocols protocols,
                    const void *extraInfo);

/**
 * \brief Error response to the remove links request.
 *
 * This is the callback used when a call to VNCAudioRouterRemoveLinks 
 * fails for some protocols. It contains the protocols that failed and the
 * failure reason.
 *
 * \param pContext The context given by the client application.
 * \param protocols The disconnected protocols. Can be a bitmask.
 * \param extraInfo Pointer to a data structure containing extra information
 *    about the protocoli that failed to unlink. The underlying structure 
 *    differs for each protocol/type of connection. 
 *     - For a Bluetooth protocol and "start client" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartClient.
 *     - For a Bluetooth protocol and "start server" option the structure is of 
 *     type VNCAudioRouterExtraInfoBtStartServer.
 *     - For an RTP Out protocol the structure is of type
 *     VNCAudioRouterExtraInfoRtpOut.
 *     - For RTP In and Voice Command protocols, the structure is of type
 *     VNCAudioRouterExtraInfoRtpIn.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterRemoveLinks
 * VNCAudioRouterExtraInfoBtStartClient, VNCAudioRouterExtraInfoBtStartServer,
 * VNCAudioRouterExtraInfoRtpOut, VNCAudioRouterExtraInfoRtpIn
 */
typedef void VNCCALL VNCAudioRouterLinkRemovalFailedCb(
                    void *pContext,
                    VNCAudioRouterProtocols protocols,
                    VNCAudioRouterError error,
                    const void *extraInfo);

/**
 * \brief Passes a log message to be logged by the client.
 *
 * This call passes a log message to the client application for logging. The
 * call can occur from any thread, but they are serialized through the use of
 * a standard synchronization mechanism (mutex/critical section). The
 * application should not call back into the router, or any of the SDKs, from
 * within this call.
 *
 * \param pContext The context given by the client application.
 * \param severity The severity of the log. Is always <= severity set in
 *       VNCAudioRouterSetLoggingSeverity.
 * \param text The text to log. Owned by the Audio Router.
 *
 * \see VNCAudioRouterCallbacks, VNCAudioRouterSetLoggingSeverity
 */
typedef void VNCCALL VNCAudioRouterLog(void *pContext,
                    vnc_int32_t severity, const char *text);

/**
 * \brief The structure that the client application needs to populate in order 
 * to receive callbacks from the Audio Router.
 *
 * The client application can leave entries for callbacks that are not needed NULL and
 * the Audio Router will not use those callbacks.
 *
 * \see VNCAudioRouterCreate, VNCAudioRouterInitialize, VNCAudioRouterSDK
 */
typedef struct
{
  /**
   * Callback notifying the application that the device is no longer
   * available.
   */
  VNCAudioRouterDeviceNotAvailableCb *deviceNotAvailableCb;
  /**
   * Callback returning the available protocols.
   */
  VNCAudioRouterProtocolsAvailableCb *protocolsAvailableCb;
  /**
   * Failure callback for the get available protocols request.
   */
  VNCAudioRouterProtocolCheckErrorCb *protocolCheckErrorCb;
  /**
   * Callback reporting the Bluetooth connection status.
   */
  VNCAudioRouterCheckBtConnectionCb *checkBtConnectionCb;
  /**
   * Successful response to the add links request.
   */
  VNCAudioRouterLinkAddedCb *linkAddedCb;
  /**
   * Error response to the add links request.
   */
  VNCAudioRouterLinkFailedCb *linkFailedCb;
  /**
   * Successful response to the remove links request.
   */
  VNCAudioRouterLinkRemovedCb *linkRemovedCb;
  /**
   * Error response to the remove links request.
   */
  VNCAudioRouterLinkRemovalFailedCb *linkRemovalFailedCb;
  /**
   * Passes a log message to be logged by the client.
   */
  VNCAudioRouterLog *log;
} VNCAudioRouterCallbacks;


/**
 * \brief Creates a new Audio Router.
 *
 * This method creates a new Audio Router to be used in conjunction with a
 * Viewer. If the Router fails to register its RFB4 extension with the Viewer
 * this call will fail.
 *
 * \param pCallbacks The callbacks structure of the client application.
 * \param callbacksSize The size of the callbacks structure.
 * \param pContext A pointer to a structure that will be reported as context
 *          within the callbacks. The memory is owned by the application, so it
 *          must ensure that it frees up the memory when no longer needed.
 * \param pViewerSdk A pointer to the viewer SDK context.
 * \param pViewer A pointer to a Viewer instance. The Viewer must be ready to
 *          get the RFB4 extension registered (an extension with the same name
 *          should not have been registered before, and the Viewer thread must 
 *          not be started).
 * \param ppRouter Pointer to the pointer that will hold the Router instance
 *          once this method completes successfully.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone if no error occurred.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - VNCAudioRouterErrorOutOfMemory if there is not enough memory.
 *  - VNCAudioRouterErrorNotLicensed if there are issues with the feature licence.
 *  - Another error if the extension can not be registered correctly.
 *
 * \see VNCAudioRouterInitialize, VNCAudioRouterCallbacks, VNCAudioRouterSDK
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterCreate(
                VNCAudioRouterCallbacks *pCallbacks,
                size_t callbacksSize,
                void *pContext,
                const VNCViewerSDK *pViewerSdk,
                VNCViewer *pViewer,
                VNCAudioRouter **ppRouter);

/**
 * \brief Switches Audio Router usage to RFB4.
 *
 * This call asks the Audio Router to start working in RFB4 extension message
 * mode. The Router will use the Viewer SDK extension message functionality to
 * communicate with the device. It will only be able to do so while the Viewer
 * session is active.
 * 
 * The Audio Router will communicate with the device that the Viewer has a
 * session with. If the session stops then VNCAudioRouterDeviceNotAvailableCb
 * will be called. The client can re-request
 * VNCAudioRouterGetAvailableProtocols. When a new Viewer session starts,
 * regardless of the device, the available protocols on the device will be
 * received.
 *
 * \param pRouter The Router that should switch to the RFB4 extension mode.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone on success.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - VNCAudioRouterErrorOutOfMemory if there is not enough memory to complete the request.
 *
 * \see VNCAudioRouterSDK
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterUseRfb4Extension(
                VNCAudioRouter *pRouter);

/**
 * \brief Switches Audio Router usage to MirrorLink.
 *
 * This call asks the Audio Router to start working in MirrorLink
 * mode. The Router will use the Discovery SDK, with the Discoverer given, to
 * communicate with the device given as the Entity. 
 * 
 * If the Entity disappears then VNCAudioRouterDeviceNotAvailableCb
 * will be called. 
 *
 * \param pRouter The Router that should switch to MirrorLink mode.
 * \param pDiscoverySdk Pointer to the Discovery SDK structure.
 * \param pDiscoverer Pointer to the MirrorLink Discoverer.
 * \param pEntity The Entity for which to establish audio connections.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone on success.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - VNCAudioRouterErrorOutOfMemory if there is not enough memory to complete the request.
 *
 * \see VNCAudioRouterSDK
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterUseMirrorLinkDevice(
                VNCAudioRouter *pRouter,
                const void *pDiscoverySdk,
                void *pDiscoverer,
                void *pEntity);

/**
 * \brief Switches Audio Router usage to \a none.
 *
 * This call asks the Audio Router to stop any usage mode. If the client
 * application temporarily has no device that it wishes to use, it can set the
 * usage to none.
 *
 * \param pRouter The Router that should switch to no usage mode.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone on success.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - VNCAudioRouterErrorOutOfMemory if there is not enough memory to complete the request.
 *
 * \see VNCAudioRouterSDK
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterUseNone(
                VNCAudioRouter *pRouter);

/**
 * \brief Asks the Router to retrieve the available protocols.
 * 
 * With this call the Router starts checking the available protocols on the
 * device. It will provide them asynchronously through the 
 * VNCAudioRouterProtocolsAvailableCb
 * callback, or report failure through VNCAudioRouterProtocolCheckErrorCb.
 *
 * If an error occurs while requested, then this will be reported to the
 * application synchronously. No callback will be received.
 * 
 * \note The client should call this method, even if the device is not available at
 * the moment, if it is interested in audio connections. As soon as the
 * device becomes available again, the client will be notified of the available
 * protocols.
 *
 * \note The usage must be set to either RFB4, or MirrorLink. The Router will
 * act according to the usage mode.
 *
 * \param pRouter The router that should check the available protocols.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone if the request is made successfully.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - An error code indicating to the application what went wrong otherwise.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterProtocolsAvailableCb,
 *   VNCAudioRouterProtocolCheckErrorCb
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterGetAvailableProtocols(
                    VNCAudioRouter *pRouter);

/**
 * \brief Retrieves some detailed information about the remote device.
 *
 * This call returns some information about the remote device. This information
 * was already retrieved through VNCAudioRouterGetAvailableProtocols.
 *
 * The information is related to the capabilties of the device for creating
 * remote audio connections.
 *
 * \param pRouter The Router from which to retrieve the information.
 * \param infoKey They name of the key to retrieve.
 *
 * \return The information retrieved from the server. NULL if not available.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterGetAvailableProtocols,
 * VNCAudioRouterRemotingInfoKeyBtStart,
 * VNCAudioRouterRemotingInfoKeyRtpOutFormat,
 * VNCAudioRouterRemotingInfoKeyRtpInFormat,
 * VNCAudioRouterRemotingInfoKeyRtpVoiceCommandFormat
 */
typedef const char *VNCCALL VNCAudioRouterGetRemotingInfo(
                    VNCAudioRouter *pRouter,
                    const char *infoKey);

/**
 * \brief Sets a configuration parameter for the audio connections.
 *
 * This call sets a configuration parameter for the audio connections. The
 * configuration must be performed before calling
 * VNCAudioRouterCheckBtConnection, VNCAudioRouterAddLinks, or
 * VNCAudioRouterRemoveLinks. It must be done for each device/usage mode.
 *
 * \param pRouter The Router that keeps the configuration.
 * \param key The key representing the parameter.
 * \param value The new value of the parameter. Set to NULL to unset it.
 *
 * \return An error code:
 *  - VNCAudioRouterErrorNone if the parameter was set correctly.
 *  - VNCAudioRouterErrorInvalidParameter if any parameters passed in are invalid.
 *  - An error code indicating to the application what went wrong otherwise.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterCheckBtConnection,
 * VNCAudioRouterAddLinks, VNCAudioRouterRemoveLinks,
 * VNCAudioRouterGetParameter, VNCAudioRouterParameterKeyBtStart,
 * VNCAudioRouterParameterKeyBtAddress, VNCAudioRouterParameterKeyBtPin,
 * VNCAudioRouterParameterKeyRtpPayloadType,
 * VNCAudioRouterParameterKeyRtpAudioIpl,
 * VNCAudioRouterParameterKeyRtpAudioMpl
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterSetParameter(
                    VNCAudioRouter *pRouter,
                    const char *key,
                    const char *value);

/**
 * \brief Retrieves a configuration parameter for the audio connections.
 *
 * This call gets a configuration parameter for the audio connections. 
 *
 * \param pRouter The Router that keeps the configuration.
 * \param key The key representing the parameter.
 *
 * \return The value for the parameter. If it's NULL then the parameter hasn't
 * been set for the given VNCAudioRouter or otherwise cannot be returned due to
 * a configuration issue.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterCheckBtConnection,
 * VNCAudioRouterAddLinks, VNCAudioRouterRemoveLinks,
 * VNCAudioRouterSetParameter, VNCAudioRouterParameterKeyBtStart,
 * VNCAudioRouterParameterKeyBtAddress, VNCAudioRouterParameterKeyBtPin,
 * VNCAudioRouterParameterKeyRtpPayloadType,
 * VNCAudioRouterParameterKeyRtpAudioIpl,
 * VNCAudioRouterParameterKeyRtpAudioMpl
 */
typedef const char *VNCCALL VNCAudioRouterGetParameter(
                    VNCAudioRouter *pRouter,
                    const char *key);

/**
 * \brief Checks the current Bluetooth connection status.
 *
 * Checks existing connections between the mobile device and the Bluetooth
 * audio device.
 *
 * This call will use the Bluetooth address configured as a parameter,
 * therefore successful configuration must be
 * accomplished before making a check request.
 *
 * The check response will be reported in the callback
 * VNCAudioRouterCheckBtConnectionCb.
 *
 * This call is not supported for MirrorLink connections and will
 * always fail with an error if used with a MirrorLink device.
 *
 * \param pRouter The Audio Router that has been already configured.
 *
 * \return VNCAudioRouterErrorNone if the request can be made, another error
 * code if it is not possible to make the request.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterSetParameter,
 * VNCAudioRouterCheckBtConnectionCb
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterCheckBtConnection(
                    VNCAudioRouter *pRouter);

/**
 * \brief Requests audio connections between the mobile device and audio devices.
 *
 * Creates connections for the protocols requested. It uses the configuration
 * provided through VNCAudioRouterSetParameter.
 *
 * It will report the successfully connected devices through
 * VNCAudioRouterLinkAddedCb. Failure is reported through the
 * VNCAudioRouterLinkFailedCb callback. For each type of protocol different
 * information can be provided in the callbacks and sometimes the client still
 * needs to take some action to ensure that audio starts streaming.
 *
 * For RTP the client needs to connect to the URL provided in the callback. For
 * Bluetooth, if the server creates the connection, then nothing needs to be
 * done. If the client is expected to create the connection then it needs to
 * connect to the address provided.
 *
 * For Bluetooth sometimes it is not possible to automatically create
 * connections. In such cases an appropiate error code will be given so the
 * user can be informed of what steps it needs to make manually.
 *
 * \param pRouter The Audio Router that has been already configured.
 * \param protocols The bitmask of the protocols that should be connected.
 *
 * \return VNCAudioRouterErrorNone if the router can start making the request,
 * otherwise an error code.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterSetParameter,
 * VNCAudioRouterLinkAddedCb, VNCAudioRouterLinkFailedCb
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterAddLinks(VNCAudioRouter *pRouter,
                    VNCAudioRouterProtocols protocols);

/**
 * \brief Removes some audio connections from the device.
 *
 * Removes audio connections on the protocols requested. The router should have
 * been configured before this is called.
 *
 * \param pRouter The Audio Router that has been already configured.
 * \param protocols The bitmask of the protocols that should be disconnected.
 *
 * \return VNCAudioRouterErrorNone if the router can start making the request,
 * otherwise an error code.
 *
 * \see VNCAudioRouterSDK, VNCAudioRouterSetParameter,
 * VNCAudioRouterLinkRemovedCb, VNCAudioRouterLinkRemovedCb
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterRemoveLinks(VNCAudioRouter *pRouter,
                    VNCAudioRouterProtocols protocols);

/**
 * \brief Destroys the Audio Router instance.
 *
 * This call destroys the Router instance. It should be called before the
 * Viewer is destroyed.
 *
 * \param pRouter The router that should be destroyed. Does nothing if the
 * router is NULL.
 *
 * \see VNCAudioRouterSDK
 */
typedef void VNCCALL VNCAudioRouterDestroy(VNCAudioRouter *pRouter);

/**
 * \brief Sets the maximum log severity.
 *
 * This call sets the maximum log severity. Once set, no logs of severity
 * greater than configured will be received.
 *
 * \param pRouter The Router to configure.
 * \param severity The maximum severity allowed. Set to -1 to avoid receiving
 *      any logs.
 */
typedef void VNCCALL VNCAudioRouterSetLoggingSeverity(VNCAudioRouter *pRouter,
                    vnc_int32_t severity);

/**
 * \brief The interface available to client application for using the 
 * audio routing available on mobile devices.
 *
 * This structure is owned by the client application, but populated by the call
 * to VNCAudioRouterInitialize.
 *
 * \see VNCAudioRouterInitialize, VNCAudioRouterCallbacks
 */
typedef struct
{
  /** Major component of the router library version number. */
  vnc_int32_t versionMajor;
  /** Minor component of the router library version number. */
  vnc_int32_t versionMinor;
  /** Patch component of the router library version number. */
  vnc_int32_t versionPatch;
  /** Build component of the router library version number. */
  vnc_int32_t versionBuild;

  /**
   * Creates a new Audio Router.
   */
  VNCAudioRouterCreate *create;
  /**
   * Switches Audio Router usage to RFB4.
   */
  VNCAudioRouterUseRfb4Extension *useRfb4Extension;
  /**
   * Switches Audio Router usage to MirrorLink.
   */
  VNCAudioRouterUseMirrorLinkDevice *useMirrorLinkDevice;
  /**
   * Switches Audio Router usage to \a none.
   */
  VNCAudioRouterUseNone *useNone;
  /**
   * Asks the Router to retrieve the available protocols.
   */
  VNCAudioRouterGetAvailableProtocols *getAvailableProtocols;
  /**
   * Retrieves some detailed information about the remote device.
   */
  VNCAudioRouterGetRemotingInfo *getRemotingInfo;
  /**
   * Sets a configuration parameter for the audio connections.
   */
  VNCAudioRouterSetParameter *setParameter;
  /**
   * Retrieves a configuration parameter for the audio connections.
   */
  VNCAudioRouterGetParameter *getParameter;
  /**
   * Checks the current Bluetooth connection status.
   */
  VNCAudioRouterCheckBtConnection *checkBtConnection;
  /**
   * Requests audio connections between the mobile device and audio devices.
   */
  VNCAudioRouterAddLinks *addLinks;
  /**
   * Removes some audio connections from the device.
   */
  VNCAudioRouterRemoveLinks *removeLinks;
  /**
   * Destroys the Audio Router instance.
   */
  VNCAudioRouterDestroy *destroy;
  /**
   * Sets the maximum log severity.
   */
  VNCAudioRouterSetLoggingSeverity *setLoggingSeverity;
} VNCAudioRouterSDK;


/**
 * \brief The type of the sole entry point to the Audio Router SDK.
 *
 * This is the type of the sole entry point into the Audio Router 
 * SDK DLL or shared object.
 *
 * \param pSDK A pointer to the structure that gets populated with the calls to
 *         the SDK.
 * \param sdkSize The size of the structure that gets populated with the calls
 *         to the SDK.
 *
 * \return An error representing the outcome of the initialization.
 *
 * \see VNCAudioRouterSDK
 */
typedef VNCAudioRouterError VNCCALL VNCAudioRouterInitializeType(
                            VNCAudioRouterSDK *pSDK,
                            size_t sdkSize);

/**
 * \brief The sole entry point to the Audio Router SDK.
 *
 * This is the sole entry point into the Audio Router 
 * SDK DLL or shared object.
 *
 * \param pSDK A pointer to the structure that gets populated with the calls to
 *         the SDK.
 * \param sdkSize The size of the structure that gets populated with the calls
 *         to the SDK.
 *
 * \return An error representing the outcome of the initialization.
 *
 * \see VNCAudioRouterSDK
 */
VNCDLLIMPORT VNCAudioRouterError VNCCALL VNCAudioRouterInitialize(
                            VNCAudioRouterSDK *pSDK,
                            size_t sdkSize);

#ifdef __cplusplus
}
#endif

#endif /* !defined (__VNCAUDIOROUTER_H__) */

